--========================================================--
-- Unseasonal Weather - DynamicRadio Channel + Broadcast
-- Minimal bridge to emit real broadcasts on 92.7 MHz.
--========================================================--

if not UW_RF then UW_RF = {} end

local UW_RF_UUID = "UW-RF-001"
local UW_RF_FREQ = 92700 -- 92.7 MHz

local function UW_RF_GetWorldHours()
    local gt = getGameTime and getGameTime()
    return gt and gt:getWorldAgeHours() or 0
end

local function UW_RF_EnsureChannel(_scriptManager)
    if not DynamicRadio then return end
    DynamicRadio.channels = DynamicRadio.channels or {}
    DynamicRadio.cache    = DynamicRadio.cache or {}

    -- If the cache already has it, return it.
    if DynamicRadio.cache[UW_RF_UUID] then
        return DynamicRadio.cache[UW_RF_UUID]
    end

    -- Make sure the channel exists in the list (avoid duplicates)
    local foundInList = false
    for _, ch in ipairs(DynamicRadio.channels) do
        if ch.uuid == UW_RF_UUID then
            foundInList = true
            break
        end
    end
    if not foundInList then
        table.insert(DynamicRadio.channels, {
            name     = "Unseasonal Weather Network",
            freq     = UW_RF_FREQ,
            category = "Emergency",
            uuid     = UW_RF_UUID,
            register = true,
        })
    end

    -- Register immediately with provided scriptManager (server event) or create directly if we can.
    if DynamicRadioChannel and DynamicRadioChannel.new then
        local cat = ChannelCategory and ChannelCategory.Emergency or ChannelCategory.Other
        local dynamicChannel = DynamicRadioChannel.new("Unseasonal Weather Network", UW_RF_FREQ, cat, UW_RF_UUID)
        if dynamicChannel then
            if dynamicChannel.setAirCounterMultiplier then
                dynamicChannel:setAirCounterMultiplier(1.0)
            end
            if _scriptManager and _scriptManager.AddChannel then
                _scriptManager:AddChannel(dynamicChannel, false)
            end
            DynamicRadio.cache[UW_RF_UUID] = dynamicChannel
            return dynamicChannel
        end
    end

    return nil
end

local function UW_RF_RegisterChannel(_scriptManager, _isNewGame)
    UW_RF_EnsureChannel(_scriptManager)
end

local function UW_RF_GetChannel()
    return UW_RF_EnsureChannel()
end

function UW_RF_EmitBroadcast(lines)
    if not lines or #lines == 0 then
        print("[UW_RF] No lines to broadcast.")
        return
    end

    local channel = UW_RF_GetChannel()
    if not channel then
        print("[UW_RF] Channel UW-RF-001 not ready.")
        return
    end

    -- Usar -1/-1 para que el juego gestione la ventana de emisión de forma nativa (como el canal vanilla).
    local bc = RadioBroadCast.new("UW Forecast - ID:" .. tostring(ZombRand(1, 99999)), -1, -1)

    for _, line in ipairs(lines) do
        if line then
            if line.setCodes then line:setCodes(false) end
            if line.setPriority then line:setPriority(5) end
            if line.setLoop then line:setLoop(false) end
            bc:AddRadioLine(line)
        end
    end

    if bc.getRadioLines and bc:getRadioLines() and bc:getRadioLines():isEmpty() then
        print("[UW_RF] WARN: Broadcast has zero lines, skipping emit.")
        return
    end

    channel:setAiringBroadcast(bc)
    print(string.format("[UW_RF] Broadcast emitted (%d lines) on %s", #lines, UW_RF_UUID))
end

function UW_RF_DebugBroadcast()
    UW_RF_EmitBroadcast({
        RadioLine.new("UW Debug Forecast Line 1", 150,150,150),
        RadioLine.new("UW Debug Forecast Line 2", 150,150,150),
    })
end

-- Debug rápido: emite texto plano sin keys ni scheduler.
function UW_RF_DebugQuick(text)
    local lineText = text or "UW_RF Debug Quick Line"
    local line = RadioLine and RadioLine.new and RadioLine.new(lineText, 150,150,150) or nil
    UW_RF_EmitBroadcast(line and { line } or { })
end

-- Debug por key de traduccion: usa UW_RF_GetLinesForKey
-- Uso: UW_RF_DebugKey("UW.Radio.Forecast.False.07")
-- Si no se pasa key, usa una por defecto.
function UW_RF_DebugKey(key)
    local useKey = key or "UW.Radio.Forecast.False.07"
    local lines = UW_RF_GetLinesForKey and UW_RF_GetLinesForKey(useKey)
    if not lines or #lines == 0 then
        print("[UW_RF][DEBUG] No hay lineas para key " .. tostring(useKey))
        return
    end
    UW_RF_EmitBroadcast(lines)
end

-- Insertar el canal lo antes posible en la lista global
UW_RF_EnsureChannel()

Events.OnLoadRadioScripts.Add(UW_RF_RegisterChannel)
Events.OnGameStart.Add(UW_RF_EnsureChannel)
Events.OnInitWorld.Add(UW_RF_EnsureChannel)
